#include "ToRORd_fkatp_endo.h"
#include <stddef.h>
#include <stdint.h>

__constant__  size_t pitch;
__constant__  real abstol;
__constant__  real reltol;
__constant__  real max_dt;
__constant__  real min_dt;
__constant__  uint8_t use_adpt;

size_t pitch_h;

extern "C" SET_ODE_INITIAL_CONDITIONS_GPU(set_model_initial_conditions_gpu) {

    uint8_t use_adpt_h = (uint8_t)solver->adaptive;

    check_cuda_error(cudaMemcpyToSymbol(use_adpt, &use_adpt_h, sizeof(uint8_t)));
    log_info("Using ToRORd_fkatp_endo GPU model\n");

    uint32_t num_volumes = solver->original_num_cells;

    if(use_adpt_h) {
        real reltol_h = solver->rel_tol;
        real abstol_h = solver->abs_tol;
        real max_dt_h = solver->max_dt;
        real min_dt_h = solver->min_dt;

        check_cuda_error(cudaMemcpyToSymbol(reltol, &reltol_h, sizeof(real)));
        check_cuda_error(cudaMemcpyToSymbol(abstol, &abstol_h, sizeof(real)));
        check_cuda_error(cudaMemcpyToSymbol(max_dt, &max_dt_h, sizeof(real)));
        check_cuda_error(cudaMemcpyToSymbol(min_dt, &min_dt_h, sizeof(real)));
        log_info("Using Adaptive Euler model to solve the ODEs\n");
    } else {
        log_info("Using Euler model to solve the ODEs\n");
    }

    // execution configuration
    const int GRID = (num_volumes + BLOCK_SIZE - 1) / BLOCK_SIZE;

    size_t size = num_volumes * sizeof(real);

    if(use_adpt_h)
        check_cuda_error(cudaMallocPitch((void **)&(solver->sv), &pitch_h, size, (size_t)NEQ + 3));
    else
        check_cuda_error(cudaMallocPitch((void **)&(solver->sv), &pitch_h, size, (size_t)NEQ));

    check_cuda_error(cudaMemcpyToSymbol(pitch, &pitch_h, sizeof(size_t)));

    real *ischFactor_device;
    real *ischFactor = NULL;

    real *APEXB_device;
    real *APEXB = NULL;
	
	int *HCMRE_device;
	int *HCMRE = NULL;
	
	int *CELLT_device;
	int *CELLT = NULL;

    real ischFactor_size = num_volumes*sizeof(real);
	real APEXB_size = num_volumes*sizeof(real);
	real HCMRE_size = num_volumes * sizeof(int);
	real CELLT_size = num_volumes * sizeof(int);

    struct extra_data_for_HCM* extra_data_from_cpu = (struct extra_data_for_HCM*)solver->ode_extra_data;
    bool deallocate = false;
	
    if(solver->ode_extra_data) {
		ischFactor = extra_data_from_cpu->ISCH;  
        HCMRE = extra_data_from_cpu->HCMRE;  
		CELLT = extra_data_from_cpu->CELLT;  
	APEXB = extra_data_from_cpu->APEXB;  
    }
    else {
        ischFactor = (real*) malloc(ischFactor_size);
		HCMRE = (int*) malloc(HCMRE_size);
		CELLT = (int*) malloc(CELLT_size);
	APEXB = (real*) malloc(APEXB_size);
		
        for(uint64_t i = 0; i < num_volumes; i++) {
            ischFactor[i] = 1.0;
			HCMRE[i] = 1.0; 
			CELLT[i] = 1.0;
		APEXB[i] = 1.0;
        }
        deallocate = true;
    }

	check_cuda_error(cudaMalloc((void **) &HCMRE_device, HCMRE_size));
    check_cuda_error(cudaMemcpy(HCMRE_device, HCMRE, HCMRE_size, cudaMemcpyHostToDevice));
	
	check_cuda_error(cudaMalloc((void **) &CELLT_device, CELLT_size));
    check_cuda_error(cudaMemcpy(CELLT_device, CELLT, CELLT_size, cudaMemcpyHostToDevice));

    check_cuda_error(cudaMalloc((void **) &ischFactor_device, ischFactor_size));
    check_cuda_error(cudaMemcpy(ischFactor_device, ischFactor, ischFactor_size, cudaMemcpyHostToDevice));

check_cuda_error(cudaMalloc((void **) &APEXB_device, APEXB_size));
    check_cuda_error(cudaMemcpy(APEXB_device, APEXB, APEXB_size, cudaMemcpyHostToDevice));

    kernel_set_model_initial_conditions<<<GRID, BLOCK_SIZE>>>(solver->sv, num_volumes, ischFactor_device, HCMRE_device, CELLT_device, APEXB_device);

    check_cuda_error(cudaPeekAtLastError());
    cudaDeviceSynchronize();
    return pitch_h;
	
    if(deallocate) free(ischFactor);
	if(deallocate) free(HCMRE);
	if(deallocate) free(CELLT);

if(deallocate) free(APEXB);
}

extern "C" SOLVE_MODEL_ODES(solve_model_odes_gpu) {

    size_t num_cells_to_solve = ode_solver->num_cells_to_solve;
    uint32_t * cells_to_solve = ode_solver->cells_to_solve;
    real *sv = ode_solver->sv;
    real dt = ode_solver->min_dt;
    uint32_t num_steps = ode_solver->num_steps;

    const int GRID = ((int)num_cells_to_solve + BLOCK_SIZE - 1) / BLOCK_SIZE;

    size_t stim_currents_size = sizeof(real) * num_cells_to_solve;
    size_t cells_to_solve_size = sizeof(uint32_t) * num_cells_to_solve;

    real *stims_currents_device;
    check_cuda_error(cudaMalloc((void **)&stims_currents_device, stim_currents_size));
    check_cuda_error(cudaMemcpy(stims_currents_device, stim_currents, stim_currents_size, cudaMemcpyHostToDevice));

    uint32_t *cells_to_solve_device = NULL;
    if(cells_to_solve != NULL) {
        check_cuda_error(cudaMalloc((void **)&cells_to_solve_device, cells_to_solve_size));
        check_cuda_error(
            cudaMemcpy(cells_to_solve_device, cells_to_solve, cells_to_solve_size, cudaMemcpyHostToDevice));
    }
	
	real *ischFactor_device;
    real *ischFactor = NULL;

real *APEXB_device;
    real *APEXB = NULL;
	
	int *HCMRE_device;
    int *HCMRE = NULL;
	
	int *CELLT_device;
    int *CELLT = NULL;
	
	real *extra_parameters_device;
	int num_extra_parameters = 4;
	size_t extra_parameters_size = num_extra_parameters * sizeof(real);
    real extra_par[num_extra_parameters];
	
    real ischFactor_size = num_cells_to_solve*sizeof(real);
	real HCMRE_size = num_cells_to_solve*sizeof(int);
	real CELLT_size = num_cells_to_solve*sizeof(int);
	real APEXB_size = num_cells_to_solve*sizeof(real);

    struct extra_data_for_HCM* extra_data_from_cpu = (struct extra_data_for_HCM*)ode_solver->ode_extra_data;
    bool deallocate = false;
	
    if(ode_solver->ode_extra_data) {
		ischFactor = extra_data_from_cpu->ISCH;
        HCMRE = extra_data_from_cpu->HCMRE;
		CELLT = extra_data_from_cpu->CELLT;
	APEXB = extra_data_from_cpu->APEXB;
        extra_par[0] = extra_data_from_cpu->INaFactor;
        extra_par[1] = extra_data_from_cpu->ICaLFactor;
        extra_par[2] = extra_data_from_cpu->Ko;
        extra_par[3] = extra_data_from_cpu->f;
    }
    else {
		extra_par[0] = 1.0f;
        extra_par[1] = 1.0f;
        extra_par[2] = 5.0f;
        extra_par[3] = 0.0f;

        ischFactor = (real*) malloc(ischFactor_size);
		HCMRE = (int*) malloc(HCMRE_size);
		CELLT = (int*) malloc(CELLT_size);
	APEXB = (real*) malloc(APEXB_size);

        for(uint64_t i = 0; i < num_cells_to_solve; i++) {
            ischFactor[i] = 1.0;
			HCMRE[i] = 1.0;
			CELLT[i] = 1.0;
		APEXB[i] = 1.0;
        }
        deallocate = true;
    }


	check_cuda_error(cudaMalloc((void **) &extra_parameters_device, extra_parameters_size));
    check_cuda_error(cudaMemcpy(extra_parameters_device, extra_par, extra_parameters_size, cudaMemcpyHostToDevice));
	
    check_cuda_error(cudaMalloc((void **) &ischFactor_device, ischFactor_size));
	check_cuda_error(cudaMalloc((void **) &APEXB_device, APEXB_size));
	check_cuda_error(cudaMalloc((void **) &HCMRE_device, HCMRE_size));
	check_cuda_error(cudaMalloc((void **) &CELLT_device, CELLT_size));
	
    check_cuda_error(cudaMemcpy(ischFactor_device, ischFactor, ischFactor_size, cudaMemcpyHostToDevice));
	check_cuda_error(cudaMemcpy(APEXB_device, APEXB, APEXB_size, cudaMemcpyHostToDevice));
	check_cuda_error(cudaMemcpy(HCMRE_device, HCMRE, HCMRE_size, cudaMemcpyHostToDevice));
    check_cuda_error(cudaMemcpy(CELLT_device, CELLT, CELLT_size, cudaMemcpyHostToDevice));

    solve_gpu<<<GRID, BLOCK_SIZE>>>(current_t, dt, sv, stims_currents_device, cells_to_solve_device, num_cells_to_solve,
                                    num_steps, ischFactor_device, extra_parameters_device, HCMRE_device, CELLT_device, APEXB_device);

    check_cuda_error(cudaPeekAtLastError());
	
	
	check_cuda_error(cudaFree(ischFactor_device));
	check_cuda_error(cudaFree(HCMRE_device));
	check_cuda_error(cudaFree(CELLT_device));
	check_cuda_error(cudaFree(APEXB_device));
    check_cuda_error(cudaFree(extra_parameters_device));
    check_cuda_error(cudaFree(stims_currents_device));
	
    if(cells_to_solve_device)
        check_cuda_error(cudaFree(cells_to_solve_device));

    if(deallocate) free(ischFactor);
	if(deallocate) free(HCMRE);
	if(deallocate) free(CELLT);
if(deallocate) free(APEXB);
    
}

__global__ void kernel_set_model_initial_conditions(real *sv, int num_volumes, real *ischFactor, int *HCMRE, int *CELLT, real *APEXB) {
    int threadID = blockDim.x * blockIdx.x + threadIdx.x;

    if(threadID < num_volumes) {
	
			
	
		if(ischFactor[threadID] >= 0.5 && HCMRE[threadID] == 1 && CELLT[threadID]==1){
			// ctrl_endo_base
            *((real *)((char *)sv + pitch * 0) + threadID) = -90.19133807633699;
            *((real *)((char *)sv + pitch * 1) + threadID) = 0.011013588522796658;
            *((real *)((char *)sv + pitch * 2) + threadID) = 12.574242903775106;
            *((real *)((char *)sv + pitch * 3) + threadID) = 12.574582392993651;
            *((real *)((char *)sv + pitch * 4) + threadID) = 150.19056907365757;
            *((real *)((char *)sv + pitch * 5) + threadID) = 150.19052768615055;
            *((real *)((char *)sv + pitch * 6) + threadID) = 7.44048473884991e-05;
            *((real *)((char *)sv + pitch * 7) + threadID) = 6.483395889394363e-05;
            *((real *)((char *)sv + pitch * 8) + threadID) = 1.5389478766441236;
            *((real *)((char *)sv + pitch * 9) + threadID) = 1.536694786785213;
            *((real *)((char *)sv + pitch * 10) + threadID) = 0.0005922217972758298;
            *((real *)((char *)sv + pitch * 11) + threadID) = 0.8551835450556216;
            *((real *)((char *)sv + pitch * 12) + threadID) = 0.8550390075147747;
            *((real *)((char *)sv + pitch * 13) + threadID) = 0.7152234160538272;
            *((real *)((char *)sv + pitch * 14) + threadID) = 0.8548184001780823;
            *((real *)((char *)sv + pitch * 15) + threadID) = 0.00012420967889311362;
            *((real *)((char *)sv + pitch * 16) + threadID) = 0.5708111260720297;
            *((real *)((char *)sv + pitch * 17) + threadID) = 0.32350557507938427;
            *((real *)((char *)sv + pitch * 18) + threadID) = 0.0008637254985821099;
            *((real *)((char *)sv + pitch * 19) + threadID) = 0.9996960963195797;
            *((real *)((char *)sv + pitch * 20) + threadID) = 0.6018783089331771;
            *((real *)((char *)sv + pitch * 21) + threadID) = 0.000440061775439436;
            *((real *)((char *)sv + pitch * 22) + threadID) = 0.9996961028699116;
            *((real *)((char *)sv + pitch * 23) + threadID) = 0.6665495911859022;
            *((real *)((char *)sv + pitch * 24) + threadID) = -8.224235092016106e-33;
            *((real *)((char *)sv + pitch * 25) + threadID) = 0.9999999949535402;
            *((real *)((char *)sv + pitch * 26) + threadID) = 0.9411304613101922;
            *((real *)((char *)sv + pitch * 27) + threadID) = 0.9999999949535722;
            *((real *)((char *)sv + pitch * 28) + threadID) = 0.9999028713135187;
            *((real *)((char *)sv + pitch * 29) + threadID) = 0.9999848048138822;
            *((real *)((char *)sv + pitch * 30) + threadID) = 0.9999999949520237;
            *((real *)((char *)sv + pitch * 31) + threadID) = 0.9999999949517666;
            *((real *)((char *)sv + pitch * 32) + threadID) = 0.00048599587368697127;
            *((real *)((char *)sv + pitch * 33) + threadID) = 0.000827241213556104;
            *((real *)((char *)sv + pitch * 34) + threadID) = 0.0006317222091527325;
            *((real *)((char *)sv + pitch * 35) + threadID) = 0.0007689905930744928;
            *((real *)((char *)sv + pitch * 36) + threadID) = 0.998332345738328;
            *((real *)((char *)sv + pitch * 37) + threadID) = 0.0002584734877950404;
            *((real *)((char *)sv + pitch * 38) + threadID) = 8.467676071452548e-06;
            *((real *)((char *)sv + pitch * 39) + threadID) = 0.24188639950596114;
            *((real *)((char *)sv + pitch * 40) + threadID) = 0.00015093606267520238;
            *((real *)((char *)sv + pitch * 41) + threadID) = -2.5163635434251287e-25;
            *((real *)((char *)sv + pitch * 42) + threadID) = 2.247386575256085e-23;		
		}
		else if(ischFactor[threadID] >= 0.5 && HCMRE[threadID] == 1 && CELLT[threadID]==3){
			// ctrl_epi_base
            *((real *)((char *)sv + pitch * 0) + threadID) = -90.29562273303094;
            *((real *)((char *)sv + pitch * 1) + threadID) = 0.012784049848454564;
            *((real *)((char *)sv + pitch * 2) + threadID) = 13.03631639329731;
            *((real *)((char *)sv + pitch * 3) + threadID) = 13.036625516039688;
            *((real *)((char *)sv + pitch * 4) + threadID) = 149.7917693533286;
            *((real *)((char *)sv + pitch * 5) + threadID) = 149.7917261804706;
            *((real *)((char *)sv + pitch * 6) + threadID) = 6.604146708533937e-05;
            *((real *)((char *)sv + pitch * 7) + threadID) = 5.746289035069578e-05;
            *((real *)((char *)sv + pitch * 8) + threadID) = 1.81582803080698;
            *((real *)((char *)sv + pitch * 9) + threadID) = 1.8140440062594907;
            *((real *)((char *)sv + pitch * 10) + threadID) = 0.0005790233170247294;
            *((real *)((char *)sv + pitch * 11) + threadID) = 0.8569848836871327;
            *((real *)((char *)sv + pitch * 12) + threadID) = 0.8569231048004134;
            *((real *)((char *)sv + pitch * 13) + threadID) = 0.7183179760032753;
            *((real *)((char *)sv + pitch * 14) + threadID) = 0.8567967858489237;
            *((real *)((char *)sv + pitch * 15) + threadID) = 0.00012177347912898773;
            *((real *)((char *)sv + pitch * 16) + threadID) = 0.5772232309803716;
            *((real *)((char *)sv + pitch * 17) + threadID) = 0.3345216752311716;
            *((real *)((char *)sv + pitch * 18) + threadID) = 0.0008576712169200143;
            *((real *)((char *)sv + pitch * 19) + threadID) = 0.9997016360831356;
            *((real *)((char *)sv + pitch * 20) + threadID) = 0.9997005267230714;
            *((real *)((char *)sv + pitch * 21) + threadID) = 0.00043697586475993384;
            *((real *)((char *)sv + pitch * 22) + threadID) = 0.9997016362110346;
            *((real *)((char *)sv + pitch * 23) + threadID) = 0.9997013993567639;
            *((real *)((char *)sv + pitch * 24) + threadID) = -3.615429697531542e-34;
            *((real *)((char *)sv + pitch * 25) + threadID) = 0.9999999950946523;
            *((real *)((char *)sv + pitch * 26) + threadID) = 0.949164823104162;
            *((real *)((char *)sv + pitch * 27) + threadID) = 0.9999999950946641;
            *((real *)((char *)sv + pitch * 28) + threadID) = 0.9999339321039371;
            *((real *)((char *)sv + pitch * 29) + threadID) = 0.999987850083045;
            *((real *)((char *)sv + pitch * 30) + threadID) = 0.9999999950939492;
            *((real *)((char *)sv + pitch * 31) + threadID) = 0.999999995093986;
            *((real *)((char *)sv + pitch * 32) + threadID) = 0.0003042031323862904;
            *((real *)((char *)sv + pitch * 33) + threadID) = 0.000521840765499422;
            *((real *)((char *)sv + pitch * 34) + threadID) = 0.000623543484446662;
            *((real *)((char *)sv + pitch * 35) + threadID) = 0.0007633393305381794;
            *((real *)((char *)sv + pitch * 36) + threadID) = 0.998401849953957;
            *((real *)((char *)sv + pitch * 37) + threadID) = 0.00020460839322048954;
            *((real *)((char *)sv + pitch * 38) + threadID) = 6.658311755552122e-06;
            *((real *)((char *)sv + pitch * 39) + threadID) = 0.22887445117377397;
            *((real *)((char *)sv + pitch * 40) + threadID) = 0.00014915168338209842;
            *((real *)((char *)sv + pitch * 41) + threadID) = 1.0683543468610166e-25;
            *((real *)((char *)sv + pitch * 42) + threadID) = -8.289146873761124e-23;			
		}
		else if(ischFactor[threadID] >= 0.5 && HCMRE[threadID] == 2 && CELLT[threadID]==1){
			// hcm_endo_base
            *((real *)((char *)sv + pitch * 0) + threadID) = -87.8309764933514;
            *((real *)((char *)sv + pitch * 1) + threadID) = 0.015426924129299652;
            *((real *)((char *)sv + pitch * 2) + threadID) = 14.791852684254422;
            *((real *)((char *)sv + pitch * 3) + threadID) = 14.79224376351163;
            *((real *)((char *)sv + pitch * 4) + threadID) = 147.04274490517386;
            *((real *)((char *)sv + pitch * 5) + threadID) = 147.04271888929918;
            *((real *)((char *)sv + pitch * 6) + threadID) = 8.061564854138384e-05;
            *((real *)((char *)sv + pitch * 7) + threadID) = 6.924217115481546e-05;
            *((real *)((char *)sv + pitch * 8) + threadID) = 1.5673711444917053;
            *((real *)((char *)sv + pitch * 9) + threadID) = 1.5566568349658165;
            *((real *)((char *)sv + pitch * 10) + threadID) = 0.0009845421888438647;
            *((real *)((char *)sv + pitch * 11) + threadID) = 0.8088903161408509;
            *((real *)((char *)sv + pitch * 12) + threadID) = 0.8064120782443271;
            *((real *)((char *)sv + pitch * 13) + threadID) = 0.639118689735458;
            *((real *)((char *)sv + pitch * 14) + threadID) = 0.8001296922768112;
            *((real *)((char *)sv + pitch * 15) + threadID) = 0.00019447396071948196;
            *((real *)((char *)sv + pitch * 16) + threadID) = 0.4544733104165169;
            *((real *)((char *)sv + pitch * 17) + threadID) = 0.20045064970736157;
            *((real *)((char *)sv + pitch * 18) + threadID) = 0.0010127520216116324;
            *((real *)((char *)sv + pitch * 19) + threadID) = 0.999540001017475;
            *((real *)((char *)sv + pitch * 20) + threadID) = 0.4582921049551983;
            *((real *)((char *)sv + pitch * 21) + threadID) = 0.0005160274440164338;
            *((real *)((char *)sv + pitch * 22) + threadID) = 0.9995400910042669;
            *((real *)((char *)sv + pitch * 23) + threadID) = 0.4984881758861471;
            *((real *)((char *)sv + pitch * 24) + threadID) = 4.542130052746222e-27;
            *((real *)((char *)sv + pitch * 25) + threadID) = 0.999999990415238;
            *((real *)((char *)sv + pitch * 26) + threadID) = 0.8689671568272143;
            *((real *)((char *)sv + pitch * 27) + threadID) = 0.9999999904158223;
            *((real *)((char *)sv + pitch * 28) + threadID) = 0.9975331047498668;
            *((real *)((char *)sv + pitch * 29) + threadID) = 0.9996234832848594;
            *((real *)((char *)sv + pitch * 30) + threadID) = 0.9999999752570458;
            *((real *)((char *)sv + pitch * 31) + threadID) = 0.9999999901012522;
            *((real *)((char *)sv + pitch * 32) + threadID) = 0.0006278022763688718;
            *((real *)((char *)sv + pitch * 33) + threadID) = 0.0011279991776972093;
            *((real *)((char *)sv + pitch * 34) + threadID) = 0.001023751442974867;
            *((real *)((char *)sv + pitch * 35) + threadID) = 0.0009061612653195337;
            *((real *)((char *)sv + pitch * 36) + threadID) = 0.9930954745574363;
            *((real *)((char *)sv + pitch * 37) + threadID) = 0.0047958072036993725;
            *((real *)((char *)sv + pitch * 38) + threadID) = 0.00017880730162265853;
            *((real *)((char *)sv + pitch * 39) + threadID) = 0.3511379609468459;
            *((real *)((char *)sv + pitch * 40) + threadID) = 0.00019725697417232377;
            *((real *)((char *)sv + pitch * 41) + threadID) = -2.275053153955557e-23;
            *((real *)((char *)sv + pitch * 42) + threadID) = -1.0288688056971914e-22;			
		}
		else if(ischFactor[threadID] >= 0.5 && HCMRE[threadID] == 2 && CELLT[threadID]==3){
			// hcm_epi_base
            *((real *)((char *)sv + pitch * 0) + threadID) = -88.23045105098176;
            *((real *)((char *)sv + pitch * 1) + threadID) = 0.019215756621552258;
            *((real *)((char *)sv + pitch * 2) + threadID) = 15.003462913935756;
            *((real *)((char *)sv + pitch * 3) + threadID) = 15.003814273812024;
            *((real *)((char *)sv + pitch * 4) + threadID) = 146.86717585249272;
            *((real *)((char *)sv + pitch * 5) + threadID) = 146.8671533501239;
            *((real *)((char *)sv + pitch * 6) + threadID) = 7.22522736663197e-05;
            *((real *)((char *)sv + pitch * 7) + threadID) = 6.192304432773449e-05;
            *((real *)((char *)sv + pitch * 8) + threadID) = 1.963710059200374;
            *((real *)((char *)sv + pitch * 9) + threadID) = 1.9556120044977934;
            *((real *)((char *)sv + pitch * 10) + threadID) = 0.0009036271548380467;
            *((real *)((char *)sv + pitch * 11) + threadID) = 0.8175330984625158;
            *((real *)((char *)sv + pitch * 12) + threadID) = 0.816167132091892;
            *((real *)((char *)sv + pitch * 13) + threadID) = 0.6528389126783802;
            *((real *)((char *)sv + pitch * 14) + threadID) = 0.8129734910006042;
            *((real *)((char *)sv + pitch * 15) + threadID) = 0.0001802643269866145;
            *((real *)((char *)sv + pitch * 16) + threadID) = 0.4807043411954388;
            *((real *)((char *)sv + pitch * 17) + threadID) = 0.2254979264375562;
            *((real *)((char *)sv + pitch * 18) + threadID) = 0.0009858231335402045;
            *((real *)((char *)sv + pitch * 19) + threadID) = 0.9995716834383294;
            *((real *)((char *)sv + pitch * 20) + threadID) = 0.9993161610982088;
            *((real *)((char *)sv + pitch * 21) + threadID) = 0.0005022997291602321;
            *((real *)((char *)sv + pitch * 22) + threadID) = 0.9995716872335115;
            *((real *)((char *)sv + pitch * 23) + threadID) = 0.9994923401514673;
            *((real *)((char *)sv + pitch * 24) + threadID) = -6.466791953622507e-29;
            *((real *)((char *)sv + pitch * 25) + threadID) = 0.999999991412432;
            *((real *)((char *)sv + pitch * 26) + threadID) = 0.8800794272431691;
            *((real *)((char *)sv + pitch * 27) + threadID) = 0.9999999914071852;
            *((real *)((char *)sv + pitch * 28) + threadID) = 0.998225603410498;
            *((real *)((char *)sv + pitch * 29) + threadID) = 0.9997858363715622;
            *((real *)((char *)sv + pitch * 30) + threadID) = 0.999999987839176;
            *((real *)((char *)sv + pitch * 31) + threadID) = 0.9999999912285538;
            *((real *)((char *)sv + pitch * 32) + threadID) = 0.0004072150730419496;
            *((real *)((char *)sv + pitch * 33) + threadID) = 0.0007396294576697753;
            *((real *)((char *)sv + pitch * 34) + threadID) = 0.0008606426727153538;
            *((real *)((char *)sv + pitch * 35) + threadID) = 0.0008824960010902431;
            *((real *)((char *)sv + pitch * 36) + threadID) = 0.995712755529311;
            *((real *)((char *)sv + pitch * 37) + threadID) = 0.0024545342112558275;
            *((real *)((char *)sv + pitch * 38) + threadID) = 8.957317154800032e-05;
            *((real *)((char *)sv + pitch * 39) + threadID) = 0.3436090385941042;
            *((real *)((char *)sv + pitch * 40) + threadID) = 0.00018835929173426506;
            *((real *)((char *)sv + pitch * 41) + threadID) = -1.668620786523204e-23;
            *((real *)((char *)sv + pitch * 42) + threadID) = 1.5561706163139582e-22;		
		}
		else if(ischFactor[threadID] < 0.5 && HCMRE[threadID] == 1 && CELLT[threadID]==1){
			// ctrl_endo_isch
            *((real *)((char *)sv + pitch * 0) + threadID) = -81.62065215053039;
            *((real *)((char *)sv + pitch * 1) + threadID) = 0.007947954723373564;
            *((real *)((char *)sv + pitch * 2) + threadID) = 12.197521150902313;
            *((real *)((char *)sv + pitch * 3) + threadID) = 12.197825553229293;
            *((real *)((char *)sv + pitch * 4) + threadID) = 150.91685199214166;
            *((real *)((char *)sv + pitch * 5) + threadID) = 150.91680877176407;
            *((real *)((char *)sv + pitch * 6) + threadID) = 6.760421599689817e-05;
            *((real *)((char *)sv + pitch * 7) + threadID) = 5.9208909450495706e-05;
            *((real *)((char *)sv + pitch * 8) + threadID) = 1.3504886651481884;
            *((real *)((char *)sv + pitch * 9) + threadID) = 1.3478265912219485;
            *((real *)((char *)sv + pitch * 10) + threadID) = 0.003664799385029553;
            *((real *)((char *)sv + pitch * 11) + threadID) = 0.6320059285442613;
            *((real *)((char *)sv + pitch * 12) + threadID) = 0.6308714809910585;
            *((real *)((char *)sv + pitch * 13) + threadID) = 0.4014575908232011;
            *((real *)((char *)sv + pitch * 14) + threadID) = 0.6260098315618794;
            *((real *)((char *)sv + pitch * 15) + threadID) = 0.0006324698233525612;
            *((real *)((char *)sv + pitch * 16) + threadID) = 0.3048306575230871;
            *((real *)((char *)sv + pitch * 17) + threadID) = 0.14644961476019797;
            *((real *)((char *)sv + pitch * 18) + threadID) = 0.0015390136142552243;
            *((real *)((char *)sv + pitch * 19) + threadID) = 0.9986384959169108;
            *((real *)((char *)sv + pitch * 20) + threadID) = 0.5893732935084903;
            *((real *)((char *)sv + pitch * 21) + threadID) = 0.0007843761296522816;
            *((real *)((char *)sv + pitch * 22) + threadID) = 0.9986385078738376;
            *((real *)((char *)sv + pitch * 23) + threadID) = 0.6424055898260645;
            *((real *)((char *)sv + pitch * 24) + threadID) = -3.4758401835962183e-31;
            *((real *)((char *)sv + pitch * 25) + threadID) = 0.9999999487081309;
            *((real *)((char *)sv + pitch * 26) + threadID) = 0.9608427986735961;
            *((real *)((char *)sv + pitch * 27) + threadID) = 0.999999948708454;
            *((real *)((char *)sv + pitch * 28) + threadID) = 0.9999621730639572;
            *((real *)((char *)sv + pitch * 29) + threadID) = 0.9999937656361052;
            *((real *)((char *)sv + pitch * 30) + threadID) = 0.9999999487196874;
            *((real *)((char *)sv + pitch * 31) + threadID) = 0.9999999486996708;
            *((real *)((char *)sv + pitch * 32) + threadID) = 0.00034180273672807627;
            *((real *)((char *)sv + pitch * 33) + threadID) = 0.0005714077674311047;
            *((real *)((char *)sv + pitch * 34) + threadID) = 0.0011908686965300126;
            *((real *)((char *)sv + pitch * 35) + threadID) = 0.0014128912504958358;
            *((real *)((char *)sv + pitch * 36) + threadID) = 0.9959179658521218;
            *((real *)((char *)sv + pitch * 37) + threadID) = 0.0014056669323029758;
            *((real *)((char *)sv + pitch * 38) + threadID) = 7.260513228015375e-05;
            *((real *)((char *)sv + pitch * 39) + threadID) = 0.1870407672923529;
            *((real *)((char *)sv + pitch * 40) + threadID) = 0.00039392274676641775;
            *((real *)((char *)sv + pitch * 41) + threadID) = -3.7510342803353864e-21;
            *((real *)((char *)sv + pitch * 42) + threadID) = 9.928201731206876e-20;			
		}
		else if(ischFactor[threadID] < 0.5 && HCMRE[threadID] == 1 && CELLT[threadID]==3){
			// ctrl_epi_isch
           *((real *)((char *)sv + pitch * 0) + threadID) = -81.66676212221233;
            *((real *)((char *)sv + pitch * 1) + threadID) = 0.009907425422641492;
            *((real *)((char *)sv + pitch * 2) + threadID) = 12.738022802377067;
            *((real *)((char *)sv + pitch * 3) + threadID) = 12.738301924372136;
            *((real *)((char *)sv + pitch * 4) + threadID) = 150.44288629143685;
            *((real *)((char *)sv + pitch * 5) + threadID) = 150.4428395730931;
            *((real *)((char *)sv + pitch * 6) + threadID) = 5.966632863919004e-05;
            *((real *)((char *)sv + pitch * 7) + threadID) = 5.2178262619735245e-05;
            *((real *)((char *)sv + pitch * 8) + threadID) = 1.5882932616556258;
            *((real *)((char *)sv + pitch * 9) + threadID) = 1.5856428862531382;
            *((real *)((char *)sv + pitch * 10) + threadID) = 0.003629800438621011;
            *((real *)((char *)sv + pitch * 11) + threadID) = 0.633633874245031;
            *((real *)((char *)sv + pitch * 12) + threadID) = 0.6328033817255506;
            *((real *)((char *)sv + pitch * 13) + threadID) = 0.40330850416926356;
            *((real *)((char *)sv + pitch * 14) + threadID) = 0.6284411917562174;
            *((real *)((char *)sv + pitch * 15) + threadID) = 0.0006269573195453916;
            *((real *)((char *)sv + pitch * 16) + threadID) = 0.3067906525297897;
            *((real *)((char *)sv + pitch * 17) + threadID) = 0.14841496777079424;
            *((real *)((char *)sv + pitch * 18) + threadID) = 0.001534236042063269;
            *((real *)((char *)sv + pitch * 19) + threadID) = 0.9986495638219557;
            *((real *)((char *)sv + pitch * 20) + threadID) = 0.996880386027121;
            *((real *)((char *)sv + pitch * 21) + threadID) = 0.000781939349981981;
            *((real *)((char *)sv + pitch * 22) + threadID) = 0.9986495644262796;
            *((real *)((char *)sv + pitch * 23) + threadID) = 0.9977159927413921;
            *((real *)((char *)sv + pitch * 24) + threadID) = 4.563628546237839e-35;
            *((real *)((char *)sv + pitch * 25) + threadID) = 0.9999999493486411;
            *((real *)((char *)sv + pitch * 26) + threadID) = 0.9645407009706896;
            *((real *)((char *)sv + pitch * 27) + threadID) = 0.9999999493488296;
            *((real *)((char *)sv + pitch * 28) + threadID) = 0.9999692546292747;
            *((real *)((char *)sv + pitch * 29) + threadID) = 0.9999940216716962;
            *((real *)((char *)sv + pitch * 30) + threadID) = 0.9999999493848009;
            *((real *)((char *)sv + pitch * 31) + threadID) = 0.9999999493479312;
            *((real *)((char *)sv + pitch * 32) + threadID) = 0.00020897774088230446;
            *((real *)((char *)sv + pitch * 33) + threadID) = 0.0003520888202834461;
            *((real *)((char *)sv + pitch * 34) + threadID) = 0.0011821795456892867;
            *((real *)((char *)sv + pitch * 35) + threadID) = 0.0014084094536066217;
            *((real *)((char *)sv + pitch * 36) + threadID) = 0.9960460905149287;
            *((real *)((char *)sv + pitch * 37) + threadID) = 0.0012965305735057585;
            *((real *)((char *)sv + pitch * 38) + threadID) = 6.678945135133245e-05;
            *((real *)((char *)sv + pitch * 39) + threadID) = 0.18433798401074333;
            *((real *)((char *)sv + pitch * 40) + threadID) = 0.00039182275577197154;
            *((real *)((char *)sv + pitch * 41) + threadID) = -3.9799044372983657e-23;
            *((real *)((char *)sv + pitch * 42) + threadID) = 5.01408482694516e-21;		
		}
		else if(ischFactor[threadID] < 0.5 && HCMRE[threadID] == 2 && CELLT[threadID]==1){
			// hcm_endo_isch
            *((real *)((char *)sv + pitch * 0) + threadID) = -79.933216641337;
            *((real *)((char *)sv + pitch * 1) + threadID) = 0.00976520970199586;
            *((real *)((char *)sv + pitch * 2) + threadID) = 14.63864186589967;
            *((real *)((char *)sv + pitch * 3) + threadID) = 14.638999817601468;
            *((real *)((char *)sv + pitch * 4) + threadID) = 147.567455591607;
            *((real *)((char *)sv + pitch * 5) + threadID) = 147.56742282398423;
            *((real *)((char *)sv + pitch * 6) + threadID) = 7.326872401652198e-05;
            *((real *)((char *)sv + pitch * 7) + threadID) = 6.30211715831054e-05;
            *((real *)((char *)sv + pitch * 8) + threadID) = 1.3584925358282849;
            *((real *)((char *)sv + pitch * 9) + threadID) = 1.3492040297149694;
            *((real *)((char *)sv + pitch * 10) + threadID) = 0.005195485958646375;
            *((real *)((char *)sv + pitch * 11) + threadID) = 0.5703808240831332;
            *((real *)((char *)sv + pitch * 12) + threadID) = 0.5601512227218602;
            *((real *)((char *)sv + pitch * 13) + threadID) = 0.335389984164233;
            *((real *)((char *)sv + pitch * 14) + threadID) = 0.5399511537305315;
            *((real *)((char *)sv + pitch * 15) + threadID) = 0.000871273174543873;
            *((real *)((char *)sv + pitch * 16) + threadID) = 0.2507702473341479;
            *((real *)((char *)sv + pitch * 17) + threadID) = 0.10686182780332251;
            *((real *)((char *)sv + pitch * 18) + threadID) = 0.001724355483397299;
            *((real *)((char *)sv + pitch * 19) + threadID) = 0.99816985492437;
            *((real *)((char *)sv + pitch * 20) + threadID) = 0.5165634783196567;
            *((real *)((char *)sv + pitch * 21) + threadID) = 0.000878917761225078;
            *((real *)((char *)sv + pitch * 22) + threadID) = 0.9981699686590519;
            *((real *)((char *)sv + pitch * 23) + threadID) = 0.5453318649716403;
            *((real *)((char *)sv + pitch * 24) + threadID) = 1.6863669832094897e-29;
            *((real *)((char *)sv + pitch * 25) + threadID) = 0.9999999188865663;
            *((real *)((char *)sv + pitch * 26) + threadID) = 0.9256801766056046;
            *((real *)((char *)sv + pitch * 27) + threadID) = 0.9999999188983804;
            *((real *)((char *)sv + pitch * 28) + threadID) = 0.9994904107436108;
            *((real *)((char *)sv + pitch * 29) + threadID) = 0.9999587210443608;
            *((real *)((char *)sv + pitch * 30) + threadID) = 0.9999999186445953;
            *((real *)((char *)sv + pitch * 31) + threadID) = 0.9999999186628105;
            *((real *)((char *)sv + pitch * 32) + threadID) = 0.00043588717111501217;
            *((real *)((char *)sv + pitch * 33) + threadID) = 0.0007805084644301835;
            *((real *)((char *)sv + pitch * 34) + threadID) = 0.001501549446990626;
            *((real *)((char *)sv + pitch * 35) + threadID) = 0.001586972586194638;
            *((real *)((char *)sv + pitch * 36) + threadID) = 0.9912363945927555;
            *((real *)((char *)sv + pitch * 37) + threadID) = 0.005371172829561337;
            *((real *)((char *)sv + pitch * 38) + threadID) = 0.00030391212731818886;
            *((real *)((char *)sv + pitch * 39) + threadID) = 0.26390762656286587;
            *((real *)((char *)sv + pitch * 40) + threadID) = 0.00047699725554300885;
            *((real *)((char *)sv + pitch * 41) + threadID) = 2.0378028467097204e-25;
            *((real *)((char *)sv + pitch * 42) + threadID) = 6.727876239591751e-23;		
		}
		else if(ischFactor[threadID] < 0.5 && HCMRE[threadID] == 2 && CELLT[threadID]==3){
			// hcm_epi_isch
            *((real *)((char *)sv + pitch * 0) + threadID) = -80.14936438519436;
            *((real *)((char *)sv + pitch * 1) + threadID) = 0.012399718695120198;
            *((real *)((char *)sv + pitch * 2) + threadID) = 14.941038314780924;
            *((real *)((char *)sv + pitch * 3) + threadID) = 14.941362458562772;
            *((real *)((char *)sv + pitch * 4) + threadID) = 147.30328626026542;
            *((real *)((char *)sv + pitch * 5) + threadID) = 147.3032556524043;
            *((real *)((char *)sv + pitch * 6) + threadID) = 6.547249361745545e-05;
            *((real *)((char *)sv + pitch * 7) + threadID) = 5.622778575432026e-05;
            *((real *)((char *)sv + pitch * 8) + threadID) = 1.6533534642807468;
            *((real *)((char *)sv + pitch * 9) + threadID) = 1.6445902983294978;
            *((real *)((char *)sv + pitch * 10) + threadID) = 0.004969549898105153;
            *((real *)((char *)sv + pitch * 11) + threadID) = 0.5786020939846324;
            *((real *)((char *)sv + pitch * 12) + threadID) = 0.5706814253155427;
            *((real *)((char *)sv + pitch * 13) + threadID) = 0.3437672234521011;
            *((real *)((char *)sv + pitch * 14) + threadID) = 0.5532503907248523;
            *((real *)((char *)sv + pitch * 15) + threadID) = 0.000836251050523797;
            *((real *)((char *)sv + pitch * 16) + threadID) = 0.25788633538384564;
            *((real *)((char *)sv + pitch * 17) + threadID) = 0.1117988978310727;
            *((real *)((char *)sv + pitch * 18) + threadID) = 0.0016994113863785668;
            *((real *)((char *)sv + pitch * 19) + threadID) = 0.9982391266909816;
            *((real *)((char *)sv + pitch * 20) + threadID) = 0.9831067653317979;
            *((real *)((char *)sv + pitch * 21) + threadID) = 0.0008661929440231542;
            *((real *)((char *)sv + pitch * 22) + threadID) = 0.9982391389809238;
            *((real *)((char *)sv + pitch * 23) + threadID) = 0.9872332553236253;
            *((real *)((char *)sv + pitch * 24) + threadID) = -6.452582195867664e-31;
            *((real *)((char *)sv + pitch * 25) + threadID) = 0.9999999235351522;
            *((real *)((char *)sv + pitch * 26) + threadID) = 0.926490064279458;
            *((real *)((char *)sv + pitch * 27) + threadID) = 0.9999999235438211;
            *((real *)((char *)sv + pitch * 28) + threadID) = 0.999547306502614;
            *((real *)((char *)sv + pitch * 29) + threadID) = 0.9999657638945735;
            *((real *)((char *)sv + pitch * 30) + threadID) = 0.9999999233298537;
            *((real *)((char *)sv + pitch * 31) + threadID) = 0.9999999233798655;
            *((real *)((char *)sv + pitch * 32) + threadID) = 0.0002798991060333818;
            *((real *)((char *)sv + pitch * 33) + threadID) = 0.0005053181293070803;
            *((real *)((char *)sv + pitch * 34) + threadID) = 0.0014448794643745749;
            *((real *)((char *)sv + pitch * 35) + threadID) = 0.001564112354987537;
            *((real *)((char *)sv + pitch * 36) + threadID) = 0.9922708860291967;
            *((real *)((char *)sv + pitch * 37) + threadID) = 0.004470109949735952;
            *((real *)((char *)sv + pitch * 38) + threadID) = 0.0002500141693017332;
            *((real *)((char *)sv + pitch * 39) + threadID) = 0.2631863039114355;
            *((real *)((char *)sv + pitch * 40) + threadID) = 0.0004652303220307569;
            *((real *)((char *)sv + pitch * 41) + threadID) = -5.854605087393694e-24;
            *((real *)((char *)sv + pitch * 42) + threadID) = 2.684185117690017e-22;		
		}

         }

        if(use_adpt) {
            *((real *)((char *)sv + pitch * 43) + threadID) = min_dt; // dt
            *((real *)((char *)sv + pitch * 44) + threadID) = 0.0;    // time_new
            *((real *)((char *)sv + pitch * 45) + threadID) = 0.0;    // previous dt
        }
}

// Solving the model for each cell in the tissue matrix ni x nj
__global__ void solve_gpu(real cur_time, real dt, real *sv, real *stim_currents, uint32_t *cells_to_solve,
                          uint32_t num_cells_to_solve, int num_steps, real *ischFactor, real *extra_parameters, int *HCMRE, int *CELLT, real *APEXB) {
    int threadID = blockDim.x * blockIdx.x + threadIdx.x;
    int sv_id;

    // Each thread solves one cell model
    if(threadID < num_cells_to_solve) {
        if(cells_to_solve)
            sv_id = cells_to_solve[threadID];
        else
            sv_id = threadID;

        if(!use_adpt) {
            real rDY[NEQ];

            for(int n = 0; n < num_steps; ++n) {

                RHS_gpu(sv, rDY, stim_currents[threadID], sv_id, dt, ischFactor[threadID], extra_parameters, HCMRE[threadID], CELLT[threadID], APEXB[threadID]);

                for(int i = 0; i < NEQ; i++) {
                    *((real *)((char *)sv + pitch * i) + sv_id) =
                        dt * rDY[i] + *((real *)((char *)sv + pitch * i) + sv_id);
                }
            }
        } else {
            solve_forward_euler_gpu_adpt(sv, stim_currents[threadID], cur_time + max_dt, sv_id, ischFactor[threadID], extra_parameters, HCMRE[threadID], CELLT[threadID], APEXB[threadID]);
        }
    }
}

inline __device__ void solve_forward_euler_gpu_adpt(real *sv, real stim_curr, real final_time, int thread_id, real ischFactor, real *extra_parameters, int HCMRE, int CELLT, real APEXB) {

    #define DT *((real *)((char *)sv + pitch * 43) + thread_id)
    #define TIME_NEW *((real *)((char *)sv + pitch * 44) + thread_id)
    #define PREVIOUS_DT *((real *)((char *)sv + pitch * 45) + thread_id)

    real rDY[NEQ];

    real _tolerances_[NEQ];
    real _aux_tol = 0.0;
    real dt = DT;
    real time_new = TIME_NEW;
    real previous_dt = PREVIOUS_DT;

    real edos_old_aux_[NEQ];
    real edos_new_euler_[NEQ];
    real _k1__[NEQ];
    real _k2__[NEQ];
    real _k_aux__[NEQ];
    real sv_local[NEQ];

    const real _beta_safety_ = 0.8;

    const real __tiny_ = powf(abstol, 2.0f);

    // dt = ((time_new + dt) > final_time) ? (final_time - time_new) : dt;
    if(time_new + dt > final_time) {
        dt = final_time - time_new;
    }

    //#pragma unroll
    for(int i = 0; i < NEQ; i++) {
        sv_local[i] = *((real *)((char *)sv + pitch * i) + thread_id);
    }

    RHS_gpu(sv_local, rDY, stim_curr, thread_id, dt, ischFactor, extra_parameters, HCMRE, CELLT, APEXB);
    time_new += dt;

    //#pragma unroll
    for(int i = 0; i < NEQ; i++) {
        _k1__[i] = rDY[i];
    }

    int count = 0;

    int count_limit = (final_time - time_new) / min_dt;

    int aux_count_limit = count_limit + 2000000;

    if(aux_count_limit > 0) {
        count_limit = aux_count_limit;
    }

    while(1) {

        for(int i = 0; i < NEQ; i++) {
            // stores the old variables in a vector
            edos_old_aux_[i] = sv_local[i];
            // //computes euler method
            edos_new_euler_[i] = _k1__[i] * dt + edos_old_aux_[i];
            // steps ahead to compute the rk2 method
            sv_local[i] = edos_new_euler_[i];
        }

        time_new += dt;

        RHS_gpu(sv_local, rDY, stim_curr, thread_id, dt, ischFactor, extra_parameters, HCMRE, CELLT, APEXB);
        time_new -= dt; // step back

        real greatestError = 0.0, auxError = 0.0;
        //#pragma unroll
        for(int i = 0; i < NEQ; i++) {

            // stores the new evaluation
            _k2__[i] = rDY[i];
            _aux_tol = fabs(edos_new_euler_[i]) * reltol;
            _tolerances_[i] = (abstol > _aux_tol) ? abstol : _aux_tol;

            // finds the greatest error between  the steps
            auxError = fabs(((dt / 2.0) * (_k1__[i] - _k2__[i])) / _tolerances_[i]);

            greatestError = (auxError > greatestError) ? auxError : greatestError;
        }

        /// adapt the time step
        greatestError += __tiny_;
        previous_dt = dt;
        /// adapt the time step
        dt = _beta_safety_ * dt * sqrt(1.0f / greatestError);

        if(time_new + dt > final_time) {
            dt = final_time - time_new;
        }

        // it doesn't accept the solution
        if(count < count_limit && (greatestError >= 1.0f)) {
            // restore the old values to do it again
            for(int i = 0; i < NEQ; i++) {
                sv_local[i] = edos_old_aux_[i];
            }
            count++;
            // throw the results away and compute again
        } else {
            count = 0;

            // if(greatestError >=1.0) {
            //    printf("Thread //d,accepting solution with error > //lf \n", threadID, greatestError);
            //}

            // it accepts the solutions
            // int aux = (dt > max_step && max_step != 0);
            // dt = (aux) ? max_step : dt;

            if(dt < min_dt) {
                dt = min_dt;
            }

            else if(dt > max_dt && max_dt != 0) {
                dt = max_dt;
            }

            if(time_new + dt > final_time) {
                dt = final_time - time_new;
            }

            // change vectors k1 e k2 , para que k2 seja aproveitado como k1 na proxima iteração
            //#pragma unroll
            for(int i = 0; i < NEQ; i++) {
                _k_aux__[i] = _k2__[i];
                _k2__[i] = _k1__[i];
                _k1__[i] = _k_aux__[i];
            }

            // it steps the method ahead, with euler solution
            //#pragma unroll
            for(int i = 0; i < NEQ; i++) {
                sv_local[i] = edos_new_euler_[i];
            }

            // verifica se o incremento para a próxima iteração ultrapassa o tempo de salvar, q neste caso é o tempo
            // final
            if(time_new + previous_dt >= final_time) {
                // se são iguais, ja foi calculada a iteração no ultimo passo de tempo e deve-se para o laço
                // nao usar igualdade - usar esta conta, pode-se mudar a tolerância
                // printf("//d: //lf\n", threadID, fabs(final_time - time_new));
                if((fabs(final_time - time_new) < 1.0e-5)) {
                    break;
                } else if(time_new < final_time) {
                    dt = previous_dt = final_time - time_new;
                    time_new += previous_dt;
                    break;
                } else {
                    dt = previous_dt = min_dt;
                    time_new += (final_time - time_new);
                    printf("Nao era pra chegar aqui: %d: %lf\n", thread_id, final_time - time_new);
                    break;
                }
            } else {
                time_new += previous_dt;
            }
        }
    }

    //#pragma unroll
    for(int i = 0; i < NEQ; i++) {
        *((real *)((char *)sv + pitch * i) + thread_id) = sv_local[i];
    }

    DT = dt;
    TIME_NEW = time_new;
    PREVIOUS_DT = previous_dt;
}

inline __device__ void RHS_gpu(real *sv, real *rDY_, real stim_current, int threadID_, real dt, real ischFactor, real *extra_parameters, int HCMRE, int CELLT, real APEXB) {

    // State variables
    real v_old_;
    real CaMKt_old_;
    real nai_old_;
    real nass_old_;
    real ki_old_;
    real kss_old_;
    real cai_old_;
    real cass_old_;
    real cansr_old_;
    real cajsr_old_;
    real m_old_;
    real h_old_;
    real j_old_;
    real hp_old_;
    real jp_old_;
    real mL_old_;
    real hL_old_;
    real hLp_old_;
    real a_old_;
    real iF_old_;
    real iS_old_;
    real ap_old_;
    real iFp_old_;
    real iSp_old_;
    real d_old_;
    real ff_old_;
    real fs_old_;
    real fcaf_old_;
    real fcas_old_;
    real jca_old_;
    real ffp_old_;
    real fcafp_old_;
    real nca_ss_old_;
    real nca_i_old_;
    real C3_old_;
    real C2_old_;
    real C1_old_;
    real O_old_;
    real I_old_;
    real xs1_old_;
    real xs2_old_;
    real Jrel_np_old_;
    real Jrel_p_old_;

    if(use_adpt) {
        v_old_ = sv[0];
        CaMKt_old_ = sv[1];
        nai_old_ = sv[2];
        nass_old_ = sv[3];
        ki_old_ = sv[4];
        kss_old_ = sv[5];
        cai_old_ = sv[6];
        cass_old_ = sv[7];
        cansr_old_ = sv[8];
        cajsr_old_ = sv[9];
        m_old_ = sv[10];
        h_old_ = sv[11];
        j_old_ = sv[12];
        hp_old_ = sv[13];
        jp_old_ = sv[14];
        mL_old_ = sv[15];
        hL_old_ = sv[16];
        hLp_old_ = sv[17];
        a_old_ = sv[18];
        iF_old_ = sv[19];
        iS_old_ = sv[20];
        ap_old_ = sv[21];
        iFp_old_ = sv[22];
        iSp_old_ = sv[23];
        d_old_ = sv[24];
        ff_old_ = sv[25];
        fs_old_ = sv[26];
        fcaf_old_ = sv[27];
        fcas_old_ = sv[28];
        jca_old_ = sv[29];
        ffp_old_ = sv[30];
        fcafp_old_ = sv[31];
        nca_ss_old_ = sv[32];
        nca_i_old_ = sv[33];
        C3_old_ = sv[34];
        C2_old_ = sv[35];
        C1_old_ = sv[36];
        O_old_ = sv[37];
        I_old_ = sv[38];
        xs1_old_ = sv[39];
        xs2_old_ = sv[40];
        Jrel_np_old_ = sv[41];
        Jrel_p_old_ = sv[42];
    } else {
        //    //State variables
        v_old_ = *((real *)((char *)sv + pitch * 0) + threadID_);
        CaMKt_old_ = *((real *)((char *)sv + pitch * 1) + threadID_);
        nai_old_ = *((real *)((char *)sv + pitch * 2) + threadID_);
        nass_old_ = *((real *)((char *)sv + pitch * 3) + threadID_);
        ki_old_ = *((real *)((char *)sv + pitch * 4) + threadID_);
        kss_old_ = *((real *)((char *)sv + pitch * 5) + threadID_);
        cai_old_ = *((real *)((char *)sv + pitch * 6) + threadID_);
        cass_old_ = *((real *)((char *)sv + pitch * 7) + threadID_);
        cansr_old_ = *((real *)((char *)sv + pitch * 8) + threadID_);
        cajsr_old_ = *((real *)((char *)sv + pitch * 9) + threadID_);
        m_old_ = *((real *)((char *)sv + pitch * 10) + threadID_);
        h_old_ = *((real *)((char *)sv + pitch * 11) + threadID_);
        j_old_ = *((real *)((char *)sv + pitch * 12) + threadID_);
        hp_old_ = *((real *)((char *)sv + pitch * 13) + threadID_);
        jp_old_ = *((real *)((char *)sv + pitch * 14) + threadID_);
        mL_old_ = *((real *)((char *)sv + pitch * 15) + threadID_);
        hL_old_ = *((real *)((char *)sv + pitch * 16) + threadID_);
        hLp_old_ = *((real *)((char *)sv + pitch * 17) + threadID_);
        a_old_ = *((real *)((char *)sv + pitch * 18) + threadID_);
        iF_old_ = *((real *)((char *)sv + pitch * 19) + threadID_);
        iS_old_ = *((real *)((char *)sv + pitch * 20) + threadID_);
        ap_old_ = *((real *)((char *)sv + pitch * 21) + threadID_);
        iFp_old_ = *((real *)((char *)sv + pitch * 22) + threadID_);
        iSp_old_ = *((real *)((char *)sv + pitch * 23) + threadID_);
        d_old_ = *((real *)((char *)sv + pitch * 24) + threadID_);
        ff_old_ = *((real *)((char *)sv + pitch * 25) + threadID_);
        fs_old_ = *((real *)((char *)sv + pitch * 26) + threadID_);
        fcaf_old_ = *((real *)((char *)sv + pitch * 27) + threadID_);
        fcas_old_ = *((real *)((char *)sv + pitch * 28) + threadID_);
        jca_old_ = *((real *)((char *)sv + pitch * 29) + threadID_);
        ffp_old_ = *((real *)((char *)sv + pitch * 30) + threadID_);
        fcafp_old_ = *((real *)((char *)sv + pitch * 31) + threadID_);
        nca_ss_old_ = *((real *)((char *)sv + pitch * 32) + threadID_);
        nca_i_old_ = *((real *)((char *)sv + pitch * 33) + threadID_);
        C3_old_ = *((real *)((char *)sv + pitch * 34) + threadID_);
        C2_old_ = *((real *)((char *)sv + pitch * 35) + threadID_);
        C1_old_ = *((real *)((char *)sv + pitch * 36) + threadID_);
        O_old_ = *((real *)((char *)sv + pitch * 37) + threadID_);
        I_old_ = *((real *)((char *)sv + pitch * 38) + threadID_);
        xs1_old_ = *((real *)((char *)sv + pitch * 39) + threadID_);
        xs2_old_ = *((real *)((char *)sv + pitch * 40) + threadID_);
        Jrel_np_old_ = *((real *)((char *)sv + pitch * 41) + threadID_);
        Jrel_p_old_ = *((real *)((char *)sv + pitch * 42) + threadID_);
    }
	
	//printf("mapping %lf ", mapping);

    #include "ToROrd_common.inc.c"
}